<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Guru;
use App\Models\Siswa;
use App\Models\TahunAjaran;
use App\Models\Semester;
use App\Models\Mapel;
use App\Models\Kelas;
use App\Models\KelasMapel;
use App\Models\Nilai;
use App\Models\SiswaKelas;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use App\Exports\SiswaKelasExport;
use App\Exports\GuruMapelExport;
use App\Exports\GuruExport;
use App\Imports\GuruImport;
use App\Imports\SiswaImport;
use App\Imports\KelasImport;
use App\Imports\GuruMapelImport;
use App\Imports\SiswaKelasImport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use App\Exports\SiswaExport;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;


class AdminController extends Controller
{

public function index() 
{
    // 1. Tahun ajaran terbaru
    $tahunAjaranTerbaru = TahunAjaran::orderByDesc('tahun_awal')->first();
    
    // Jika tidak ada tahun ajaran, return dengan data kosong
    if (!$tahunAjaranTerbaru) {
        return view('backend.admin.index', [
            'siswaAktifCount' => 0,
            'guruCount' => 0,
            'kelasCount' => 0,
            'tahunAjaranTerbaru' => null
        ]);
    }

    // 2. Hitung siswa aktif (punya kelas di tahun ajaran terbaru)
    $kelasIds = Kelas::whereHas('semester', fn($q) => $q->where('tahun_ajaran_id', $tahunAjaranTerbaru->id))
        ->pluck('id');
    
    // Jika tidak ada kelas di tahun ajaran terbaru, siswa aktif = 0
    $siswaAktifCount = 0;
    if ($kelasIds->isNotEmpty()) {
        $siswaAktifCount = SiswaKelas::whereIn('kelas_id', $kelasIds)
            ->distinct('siswa_id')->count('siswa_id');
    }

    // 3. Hitung guru yang mengajar (dari tabel kelas_mapel) di tahun ajaran terbaru
$kelasIds = Kelas::whereHas('semester', fn($q) =>
    $q->where('tahun_ajaran_id', $tahunAjaranTerbaru->id)
)->pluck('id');

$guruCount = 0;
if ($kelasIds->isNotEmpty()) {
    $guruCount = KelasMapel::whereIn('kelas_id', $kelasIds)
        ->pluck('guru_id')   // ambil semua guru yang ngajar
        ->unique()           // buang duplikat
        ->count();           // hitung jumlahnya
}


    // 4. Hitung total kelas (7–9) tahun ajaran terbaru, hanya hitung sekali per kelas
    $kelasCount = Kelas::whereBetween('nama', [7, 9])
        ->whereHas('semester', fn($q) => $q->where('tahun_ajaran_id', $tahunAjaranTerbaru->id))
        ->distinct('id')->count('id');

    return view('backend.admin.index', compact(
        'siswaAktifCount', 'guruCount', 'kelasCount', 'tahunAjaranTerbaru'
    ));

}

public function data_siswa()
{
    $tahunAjaranTerbaru = TahunAjaran::orderByDesc('tahun_awal')->first();
    
    if (!$tahunAjaranTerbaru) {
        $siswa = Siswa::with('user')
            ->orderByDesc('created_at')
            ->get();
        return view('backend.admin.data_siswa', compact('siswa'));
    }
    
    // Mendapatkan ID kelas yang aktif di tahun ajaran terbaru
    $kelasIds = Kelas::whereHas('semester', function ($q) use ($tahunAjaranTerbaru) {
        $q->where('tahun_ajaran_id', $tahunAjaranTerbaru->id);
    })->pluck('id');
    
    $siswaYangSudahPunyaKelas = SiswaKelas::whereIn('kelas_id', $kelasIds)->pluck('siswa_id');
    
    $siswa = Siswa::with('user')
        ->whereNotIn('siswa_id', $siswaYangSudahPunyaKelas)
        ->orderByDesc('created_at')
        ->get();
        
    return view('backend.admin.data_siswa', compact('siswa'));
}

    public function updateSemuaSiswa(Request $request, Siswa $siswa)
{
    $request->validate([
        'siswa_id'        => 'required|exists:users,id',
        'name'            => 'required',
        'email'           => 'required|email|unique:users,email,' . $request->siswa_id,
        'nisn'            => 'required|unique:siswa,nisn,' . $siswa->id,
        'alamat'          => 'required',
        'tempat_lahir'    => 'required',
        'tanggal_lahir'   => 'required|date',
        'jenis_kelamin'   => 'required',
        'agama'           => 'required',
        'no_hp'           => 'required',
        'sekolah_asal'    => 'required',
        'tempat_tinggal'         => 'required',
        'nik'                    => 'required|numeric',
        'no_kk'                  => 'required|numeric',
        'no_reg_akta'            => 'required',
        'kewarganegaraan'        => 'required',
        'berkebutuhan_khusus'    => 'nullable|string',
        'anak_ke'                => 'required|numeric',
        'jumlah_saudara_kandung' => 'required|numeric',
    ]);

    \DB::beginTransaction();
    try {
        $user = User::findOrFail($request->siswa_id);
        $user->name = $request->name;
        $user->email = $request->email;
        $user->role = 'siswa'; // Atau $request->role jika dinamis

        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }

        $user->save();

        $siswa->update([
            'siswa_id'        => $user->id,
            'nisn'            => $request->nisn,
            'alamat'          => $request->alamat,
            'tempat_lahir'    => $request->tempat_lahir,
            'tanggal_lahir'   => $request->tanggal_lahir,
            'jenis_kelamin'   => $request->jenis_kelamin,
            'agama'           => $request->agama,
            'no_hp'           => $request->no_hp,
            'sekolah_asal'    => $request->sekolah_asal,
            'tempat_tinggal'        => $request->tempat_tinggal,
            'nik'                   => $request->nik,
            'no_kk'                 => $request->no_kk,
            'no_reg_akta'           => $request->no_reg_akta,
            'kewarganegaraan'       => $request->kewarganegaraan,
            'berkebutuhan_khusus'   => $request->berkebutuhan_khusus,
            'anak_ke'               => $request->anak_ke,
            'jumlah_saudara_kandung'=> $request->jumlah_saudara_kandung,
        ]);

        \DB::commit();
        return redirect()->route('backend.admin.data_siswa')->with('success', 'Data siswa berhasil diperbarui.');
    } catch (\Exception $e) {
        \DB::rollBack();
        return back()->withErrors(['error' => 'Gagal memperbarui data: ' . $e->getMessage()]);
    }
}


    public function deleteSemuaSiswa(Siswa $siswa)
    {
        \DB::beginTransaction();
        try {
            $user = User::findOrFail($siswa->siswa_id);
            $siswa->delete(); // Hapus data siswa
            $user->delete(); // Hapus user

            \DB::commit();
            return redirect()->route('backend.admin.data_siswa')->with('success', 'Data siswa berhasil dihapus.');
        } catch (\Exception $e) {
            \DB::rollback();
            return back()->withErrors(['error' => 'Gagal menghapus data: ' . $e->getMessage()]);
        }

    }
    public function profil()
{
    $user = User::where('role', 'admin')->orderBy('created_at', 'desc')->get();
    return view('backend.admin.profil', compact('user'));
}

    public function storeProfil(Request $request)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email',
        'password' => 'required',
    ]);

    User::create([
        'name' => $request->name,
        'email' => $request->email,
        'password' => Hash::make($request->password),
        'role' => 'admin',
    ]);

    return redirect()->back()->with('success', 'Akun admin berhasil ditambahkan.');
}


  public function updateProfil(Request $request, $id)
{
    $request->validate([
        'name' => 'required|string|max:255',
        'email' => 'required|email|unique:users,email,' . $id,
        'password' => 'nullable',
    ]);

    $user = User::findOrFail($id);

    $data = [
        'name' => $request->name,
        'email' => $request->email,
        'role' => 'admin',
    ];

    if (!empty($request->password)) {
        $data['password'] = Hash::make($request->password);
    }

    $user->update($data);

    return redirect()->back()->with('success', 'Data admin berhasil diperbarui.');
}


    public function deleteProfil($id)
    {
        $user = User::findOrFail($id);
        $user->delete();

        return redirect()->back()->with('success', 'Data admin berhasil dihapus.');
    }

    // ================= GURU =================
    public function guru(Request $request)
    {
        $role = $request->query('role');

        $guru = Guru::with('user')
        ->when($role, function ($query) use ($role) {
            $query->whereHas('user', function ($q) use ($role) {
                $q->where('role', $role);
            });
        })
        ->orderBy('created_at', 'desc')
        ->get();

        // dd($guru);
        return view('backend.admin.guru', compact('guru'));
    }

    public function importGuru(Request $request)
{
    $request->validate([
        'file' => 'required|mimes:xlsx,xls,csv,tsv',
        'role' => 'required|in:guru,admin,guru_bk,kepala_sekolah',
    ]);

    try {
          Excel::import(new GuruImport($request->role), $request->file('file'));

        return redirect()->route('backend.admin.guru')->with('success', 'Import data guru berhasil.');
    } catch (\Exception $e) {
        return back()->withErrors(['error' => 'Gagal import: ' . $e->getMessage()]);
    }
}

public function exportGuru()
{
    $tahunAjaran = TahunAjaran::orderByDesc('tahun_awal')->first();
    $tahunAwal = $tahunAjaran->tahun_awal ?? now()->year;
    $tahunAkhir = $tahunAjaran->tahun_akhir ?? now()->addYear()->year;

    $fileName = "data_guru_{$tahunAwal}-{$tahunAkhir}.xlsx";

    return Excel::download(new GuruExport, $fileName);
}

    public function storeGuru(Request $request)
{
    $request->validate([
        'name'     => 'required',
        'email'    => 'required|email|unique:users,email',
        'password' => 'required',
        'no_hp'    => ['required', 'regex:/^\+62\d{9,}$/'],
        'role'     => 'required|in:guru,guru_bk,kepala_sekolah',
        'nip'      => 'required|unique:guru,nip',
    ], [
        'no_hp.regex' => 'Nomor HP harus diawali dengan +62 dan diikuti angka.',
        'nip.unique'  => 'NIP sudah terdaftar di sistem.',
        'email.unique' => 'Email sudah terdaftar di sistem.',
    ]);

    \DB::beginTransaction();
    try {
        // Simpan ke tabel user
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => \Hash::make($request->password),
            'role'     => $request->role,
        ]);

        // Simpan ke tabel guru
        Guru::create([
            'guru_id'       => $user->id,
            'nip'           => $request->nip,
            'no_hp'         => $request->no_hp,
            'jenis_kelamin' => $request->jenis_kelamin,
        ]);

        \DB::commit();
        return redirect()->route('backend.admin.guru')->with('success', 'Data guru berhasil ditambahkan.');
    } catch (\Exception $e) {
        \DB::rollback();
        return back()->withErrors(['error' => 'Gagal menyimpan data: ' . $e->getMessage()]);
    }
}


public function updateGuru(Request $request, Guru $guru)
{
    $request->validate([
        'guru_id'    => 'required|exists:users,id',
        'name'       => 'required',
        'email'      => 'required|email|unique:users,email,' . $guru->guru_id,
        'nip'        => 'required|unique:guru,nip,' . $guru->id,
        'no_hp'      => ['required', 'regex:/^\+62\d{9,}$/'],
        'jenis_kelamin' => 'required',
        'password'   => 'nullable',
        'role'       => 'required|in:guru,guru_bk,kepala_sekolah',
    ], [
        'no_hp.regex' => 'Nomor HP harus diawali dengan +62 dan diikuti angka.',
        'nip.unique'  => 'NIP sudah terdaftar di sistem.',
        'email.unique'=> 'Email sudah terdaftar di sistem.',
    ]);

    \DB::beginTransaction();
    try {
        $user = User::findOrFail($request->guru_id);
        $user->name = $request->name;
        $user->email = $request->email;
        $user->role = $request->role;

        if ($request->filled('password')) {
            $user->password = \Hash::make($request->password);
        }

        $user->save();

        $guru->update([
            'nip'           => $request->nip,
            'no_hp'         => $request->no_hp,
            'jenis_kelamin' => $request->jenis_kelamin,
        ]);

        \DB::commit();
        return redirect()->route('backend.admin.guru')->with('success', 'Data guru berhasil diperbarui.');
    } catch (\Exception $e) {
        \DB::rollBack();
        return back()->withErrors(['error' => 'Gagal memperbarui data: ' . $e->getMessage()]);
    }
}

    
    public function deleteGuru(Guru $guru)
    {
        \DB::beginTransaction();
        try {
            $user = User::findOrFail($guru->guru_id);
            $guru->delete(); // Hapus data guru
            $user->delete(); // Hapus user terkait

            \DB::commit();
            return redirect()->route('backend.admin.guru')->with('success', 'Data guru berhasil dihapus.');
        } catch (\Exception $e) {
            \DB::rollback();
            return back()->withErrors(['error' => 'Gagal menghapus data: ' . $e->getMessage()]);
        }
    }

    // ================= SISWA =================
public function siswa()
{
    $tahunAjaranTerbaru = TahunAjaran::orderByDesc('tahun_awal')->first();

    $kelasIds = Kelas::whereHas('semester', function ($q) use ($tahunAjaranTerbaru) {
        $q->where('tahun_ajaran_id', $tahunAjaranTerbaru->id);
    })->pluck('id');

    $siswaIds = SiswaKelas::whereIn('kelas_id', $kelasIds)->pluck('siswa_id');

    $siswa = Siswa::with('user')
    ->whereIn('siswa_id', $siswaIds)
    ->orderBy('created_at', 'desc')
    ->get();


    return view('backend.admin.siswa', compact('siswa'));
}

    public function importSiswa(Request $request)
{
    $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:2048'
        ], [
            'file.required' => 'File harus dipilih',
            'file.mimes' => 'Format file harus xlsx, xls, atau csv',
        ]);

        try {
            Excel::import(new SiswaImport, $request->file('file'));
            
            return redirect()->back()->with('success', 'Data siswa berhasil diimport!');
            
        } catch (\Exception $e) {
            Log::error('Import error: ' . $e->getMessage());
            
            return redirect()->back()->withErrors(['import' => $e->getMessage()]);
        }
    
}

    public function exportSiswa()
    {
        $tahun = TahunAjaran::orderByDesc('tahun_awal')->first();

        $tahunAwal  = $tahun?->tahun_awal ?? date('Y');
        $tahunAkhir = $tahun?->tahun_akhir ?? (date('Y') + 1);

        $fileName = "Data_Siswa_Aktif_{$tahunAwal}-{$tahunAkhir}.xlsx";

        return Excel::download(new SiswaExport, $fileName);
    }

public function exportSiswaKelas($kelas_id)
{
    $kelas = Kelas::with('semester.tahunAjaran')->findOrFail($kelas_id);

    $namaKelas   = $kelas->nama;
    $semester    = $kelas->semester->semester ?? 'x';
    $tahunAwal   = $kelas->semester->tahunAjaran->tahun_awal ?? 'xxxx';
    $tahunAkhir  = $kelas->semester->tahunAjaran->tahun_akhir ?? 'yyyy';

    $fileName = "data_siswa-kelas_{$namaKelas}_semester-{$semester}_{$tahunAwal}-{$tahunAkhir}.xlsx";

    return Excel::download(new SiswaKelasExport($kelas_id), $fileName);
}

public function storeSiswa(Request $request) 
{
    $request->validate([
        'name'                   => 'required',
        'email'                  => 'required|email|unique:users',
        'password'               => 'required',
        'nisn'                   => 'required|unique:siswa',
        'nipd'                   => 'required',
        'jenis_kelamin'          => 'required',
        'agama'                  => 'required',
        'tempat_lahir'           => 'required',
        'tanggal_lahir'          => 'required|date',
        'sekolah_asal'           => 'required',
        'no_hp'                  => 'required',
        'no_telepon'             => 'nullable',
        'alamat'                 => 'required',
        'jenis_tinggal'          => 'required',
        'berkebutuhan_khusus'    => 'nullable|string',
        'alat_transportasi'      => 'required',
        'anak_ke'                => 'required|numeric',
        'jumlah_saudara_kandung' => 'required|numeric',
    ]);

    \DB::beginTransaction();
    try {
        // Simpan ke tabel user
        $user = User::create([
            'name'     => $request->name,
            'email'    => $request->email,
            'password' => Hash::make($request->password),
            'role'     => 'siswa',
        ]);

        // Simpan ke tabel siswa
        Siswa::create([
            'siswa_id'               => $user->id,
            'nisn'                   => $request->nisn,
            'nipd'                   => $request->nipd,
            'jenis_kelamin'          => $request->jenis_kelamin,
            'agama'                  => $request->agama,
            'tempat_lahir'           => $request->tempat_lahir,
            'tanggal_lahir'          => $request->tanggal_lahir,
            'sekolah_asal'           => $request->sekolah_asal,
            'no_hp'                  => $request->no_hp,
            'no_telepon'             => $request->no_telepon,
            'alamat'                 => $request->alamat,
            'jenis_tinggal'          => $request->jenis_tinggal,
            'berkebutuhan_khusus'    => $request->berkebutuhan_khusus,
            'alat_transportasi'      => $request->alat_transportasi,
            'anak_ke'                => $request->anak_ke,
            'jumlah_saudara_kandung' => $request->jumlah_saudara_kandung,
        ]);

        \DB::commit();
        return redirect()->route('backend.admin.data_siswa')->with('success', 'Data Siswa berhasil ditambahkan.');
    } catch (\Exception $e) {
        \DB::rollback();
        return back()->withErrors(['error' => 'Gagal menyimpan data: ' . $e->getMessage()]);
    }
}

public function updateSiswa(Request $request, Siswa $siswa)
{
    $request->validate([
        'siswa_id'               => 'required|exists:users,id',
        'name'                   => 'required',
        'email'                  => 'required|email|unique:users,email,' . $request->siswa_id,
        'nisn'                   => 'required|unique:siswa,nisn,' . $siswa->id,
        'nipd'                   => 'required',
        'jenis_kelamin'          => 'required',
        'agama'                  => 'required',
        'tempat_lahir'           => 'required',
        'tanggal_lahir'          => 'required|date',
        'sekolah_asal'           => 'required',
        'no_hp'                  => 'required',
        'no_telepon'             => 'nullable',
        'alamat'                 => 'required',
        'jenis_tinggal'          => 'required',
        'berkebutuhan_khusus'    => 'nullable|string',
        'alat_transportasi'      => 'required',
        'anak_ke'                => 'required|numeric',
        'jumlah_saudara_kandung' => 'required|numeric',
        'rombel_saat_ini'        => 'nullable',
    ]);

    \DB::beginTransaction();
    try {
        $user = User::findOrFail($request->siswa_id);
        $user->name = $request->name;
        $user->email = $request->email;
        $user->role = 'siswa';

        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }

        $user->save();

        $siswa->update([
            'siswa_id'               => $user->id,
            'nisn'                   => $request->nisn,
            'nipd'                   => $request->nipd,
            'jenis_kelamin'          => $request->jenis_kelamin,
            'agama'                  => $request->agama,
            'tempat_lahir'           => $request->tempat_lahir,
            'tanggal_lahir'          => $request->tanggal_lahir,
            'sekolah_asal'           => $request->sekolah_asal,
            'no_hp'                  => $request->no_hp,
            'no_telepon'             => $request->no_telepon,
            'alamat'                 => $request->alamat,
            'jenis_tinggal'          => $request->jenis_tinggal,
            'berkebutuhan_khusus'    => $request->berkebutuhan_khusus,
            'alat_transportasi'      => $request->alat_transportasi,
            'anak_ke'                => $request->anak_ke,
            'jumlah_saudara_kandung' => $request->jumlah_saudara_kandung,
            'rombel_saat_ini'        => $request->rombel_saat_ini,
        ]);

        \DB::commit();
        return redirect()->route('backend.admin.siswa')->with('success', 'Data siswa berhasil diperbarui.');
    } catch (\Exception $e) {
        \DB::rollBack();
        return back()->withErrors(['error' => 'Gagal memperbarui data: ' . $e->getMessage()]);
    }
}

    public function deleteSiswa(Siswa $siswa)
    {
        \DB::beginTransaction();
        try {
            $user = User::findOrFail($siswa->siswa_id);
            $siswa->delete(); // Hapus data siswa
            $user->delete(); // Hapus user

            \DB::commit();
            return redirect()->route('backend.admin.siswa')->with('success', 'Data siswa berhasil dihapus.');
        } catch (\Exception $e) {
            \DB::rollback();
            return back()->withErrors(['error' => 'Gagal menghapus data: ' . $e->getMessage()]);
        }

    }

    public function restoreViewSiswa()
{
    $deletedSiswas = Siswa::onlyTrashed()
    ->with('user')
    ->orderBy('deleted_at', 'desc')
    ->get();

    return view('backend.admin.siswa_restore', compact('deletedSiswas'));
}

public function restoreSiswa($id)
{
    $siswa = Siswa::onlyTrashed()->where('id', $id)->firstOrFail();
    $siswa->restore();

    $user = User::onlyTrashed()->where('id', $siswa->siswa_id)->first();
    if ($user) {
        $user->restore();
    }

    return redirect()->route('backend.admin.data_siswa')->with('success', 'Siswa berhasil dipulihkan.');
}

public function deleteSiswaPermanent($id)
{
    $siswa = Siswa::onlyTrashed()->where('id', $id)->firstOrFail();

    $user = User::onlyTrashed()->where('id', $siswa->siswa_id)->first();
    if ($user) {
        $user->forceDelete();
    }

    $siswa->forceDelete();

    return redirect()->route('admin.siswa.restoreViewSiswa')->with('success', 'Siswa berhasil dihapus permanen.');
}

   public function tahun()
{
    $tahun = TahunAjaran::orderBy('created_at', 'desc')->get();
    return view('backend.admin.tahun', compact('tahun'));
}

    public function storeTahun(Request $request)
    {
        $request->validate([
            'tahun_awal' => 'required|string',
            'tahun_akhir' => 'required|string',
        ]);

        TahunAjaran::create($request->all());

        return redirect()->back()->with('success', 'Data tahun berhasil ditambahkan.');
    }

    public function updateTahun(Request $request, $id)
    {
        $request->validate([
            'tahun_awal' => 'required|string',
            'tahun_akhir' => 'required|string',
        ]);

        $tahun = TahunAjaran::findOrFail($id);
        $tahun->update($request->all());

        return redirect()->back()->with('success', 'Data tahun berhasil diperbarui.');
    }

    public function deleteTahun($id)
    {
        $tahun = TahunAjaran::findOrFail($id);
        $tahun->delete();

        return redirect()->back()->with('success', 'Data tahun berhasil dihapus.');
    }

public function semester()
{
    // Ambil semua semester, termasuk relasi tahun ajaran
    $semester = Semester::with([
        'tahunAjaran' => function ($query) {
            $query->withTrashed();
        }
    ])->orderBy('created_at', 'desc')->get();

    $tahun = TahunAjaran::withTrashed()->orderByDesc('created_at')->get();

    return view('backend.admin.semester', compact('semester', 'tahun'));
}


public function storeSemester(Request $request)
{
    $request->validate([
        'semester' => 'required|in:1,2',
        'tahun_ajaran_id' => 'required|exists:tahun_ajarans,id',
    ]);

    Semester::create([
        'semester' => $request->semester,
        'tahun_ajaran_id' => $request->tahun_ajaran_id,
    ]);

    return redirect()->back()->with('success', 'Data semester berhasil ditambahkan.');
}

public function updateSemester(Request $request, $id)
{
    $request->validate([
        'semester' => 'required|in:1,2',
        'tahun_ajaran_id' => 'required|exists:tahun_ajarans,id',
    ]);

    $semester = Semester::findOrFail($id);
    $semester->update([
        'semester' => $request->semester,
        'tahun_ajaran_id' => $request->tahun_ajaran_id,
    ]);

    return redirect()->back()->with('success', 'Data semester berhasil diperbarui.');
}

public function deleteSemester($id)
{
    $semester = Semester::findOrFail($id);
    $semester->delete();

    return redirect()->back()->with('success', 'Data semester berhasil dihapus.');
}

 public function mapel()
{
    $mapel = Mapel::orderBy('created_at', 'desc')->get();
    return view('backend.admin.mapel', compact('mapel'));
}

    public function storeMapel(Request $request)
{
    $request->validate([
        'nama' => 'required',
        'tipe' => 'required|boolean',
    ]);

    Mapel::create([
        'nama' => $request->nama,
        'tipe' => $request->tipe,
    ]);

    return redirect()->back()->with('success', 'Data Mapel berhasil ditambahkan.');
}

public function updateMapel(Request $request, $id)
{
    $request->validate([
        'nama' => 'required',
        'tipe' => 'required|boolean',
    ]);

    $mapel = Mapel::findOrFail($id);
    $mapel->update([
        'nama' => $request->nama,
        'tipe' => $request->tipe,
    ]);

    return redirect()->back()->with('success', 'Data Mapel berhasil diperbarui.');
}

public function deleteMapel($id)
{
    $mapel = Mapel::findOrFail($id);
    $mapel->delete();

    return redirect()->back()->with('success', 'Data Mapel berhasil dihapus.');
}

public function kelas()
{
    // Ambil semua data kelas beserta relasi yang mungkin sudah di-soft-delete
    $kelas = Kelas::with([
        'waliKelas' => fn ($query) => $query->withTrashed(),
        'semester' => fn ($query) =>
            $query->withTrashed()->with([
                'tahunAjaran' => fn ($subQuery) => $subQuery->withTrashed()
            ])
    ])
    ->orderByDesc('created_at')
    ->get();

    $guruAll = User::where('role', 'guru')
            ->with(['guru' => fn ($query) => $query->withTrashed()])
            ->withTrashed()
            ->latest()
            ->get();

    $guruAktif = User::where('role', 'guru')
            ->whereNull('deleted_at')
            ->whereHas('guru', fn ($query) => $query->whereNull('deleted_at'))
            ->with('guru')
            ->latest()
            ->get();



    // Ambil semua semester dengan tahun ajaran (termasuk soft-deleted)
    $semester = Semester::with([
    'tahunAjaran' => fn ($query) => $query->withTrashed()
])
->withTrashed()
->orderByDesc('created_at') // urutkan semester dari yang terbaru
->get();

    // Ambil semua tahun ajaran, termasuk yang soft-deleted
    $tahun = TahunAjaran::withTrashed()->get();

    return view('backend.admin.kelas', compact('kelas', 'guruAll', 'guruAktif', 'semester', 'tahun'));

}

public function importKelas(Request $request) 
{
    $request->validate([
        'file' => 'required|mimes:xlsx,xls'
    ]);
 
    try {
        Excel::import(new KelasImport, $request->file('file'));
        return back()->with('success', 'Data kelas berhasil diimpor.');
    } catch (\Exception $e) {
        return back()->withErrors(['error' => 'Gagal mengimpor data: ' . $e->getMessage()]);
    }
}

public function storeKelas(Request $request)
{
    $request->validate([
        'nama' => 'required',
        'wali_kelas_id' => 'required|exists:users,id',
        'semester_id' => 'required|exists:semesters,id',
        'tahun_ajaran_id' => 'required|exists:tahun_ajarans,id',
    ]);

    // Cek duplikat kombinasi nama + semester + tahun ajaran
    $kelasExist = Kelas::where('nama', $request->nama)
        ->where('semester_id', $request->semester_id)
        ->where('tahun_ajaran_id', $request->tahun_ajaran_id)
        ->exists();

    if ($kelasExist) {
        return redirect()->back()->withErrors(['nama' => 'Kelas ini sudah ada.'])->withInput();
    }

    // Cek apakah guru sudah menjadi wali kelas di tahun ajaran tersebut (abaikan semester dan nama)
    $waliSudahDipakai = Kelas::where('wali_kelas_id', $request->wali_kelas_id)
        ->where('tahun_ajaran_id', $request->tahun_ajaran_id)
        ->exists();

    if ($waliSudahDipakai) {
        return redirect()->back()->withErrors(['wali_kelas_id' => 'Guru ini sudah menjadi wali kelas di tahun ajaran yang dipilih.'])->withInput();
    }

    try {
        // Kapitalisasi huruf terakhir jika input seperti "7b"
        $nama = strtoupper(substr($request->nama, -1));
        $namaKelas = ucfirst(substr($request->nama, 0, -1)) . $nama;

        Kelas::create([
            'nama' => $namaKelas,
            'wali_kelas_id' => $request->wali_kelas_id,
            'semester_id' => $request->semester_id,
            'tahun_ajaran_id' => $request->tahun_ajaran_id,
        ]);

        return redirect()->back()->with('success', 'Data Kelas berhasil ditambahkan.');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Terjadi kesalahan saat menyimpan data.');
    }
}

public function updateKelas(Request $request, $id)
{
    $request->validate([
        'nama' => 'required',
        'wali_kelas_id' => 'required|exists:users,id',
        'semester_id' => 'required|exists:semesters,id',
        'tahun_ajaran_id' => 'required|exists:tahun_ajarans,id',
    ]);

    // Ambil data kelas yang sedang diupdate
    $kelas = Kelas::findOrFail($id);

    // Cek duplikat kombinasi data
    $kelasExist = Kelas::where('id', '!=', $id)
        ->where('nama', $request->nama)
        ->where('semester_id', $request->semester_id)
        ->where('tahun_ajaran_id', $request->tahun_ajaran_id)
        ->exists();

    if ($kelasExist) {
        return redirect()->back()->withErrors(['nama' => 'Kelas / Semester ini sudah ada.'])->withInput();
    }

    // Cek apakah guru sudah jadi wali kelas di tahun ajaran ini di kelas lain
    $waliSudahDipakai = Kelas::where('id', '!=', $id)
        ->where('nama', $request->nama)
        ->where('wali_kelas_id', $request->wali_kelas_id)
        ->where('semester_id', $request->semester_id)
        ->where('tahun_ajaran_id', $request->tahun_ajaran_id)
        ->exists();

    if ($waliSudahDipakai) {
        return redirect()->back()->withErrors(['wali_kelas_id' => 'Guru ini sudah menjadi wali kelas di tahun ajaran yang dipilih.'])->withInput();
    }

    // Kapitalisasi huruf terakhir di nama kelas
    $nama = strtoupper(substr($request->nama, -1));
    $namaKelas = ucfirst(substr($request->nama, 0, -1)) . $nama;

    try {
        $kelas->update([
            'nama' => $namaKelas,
            'wali_kelas_id' => $request->wali_kelas_id,
            'semester_id' => $request->semester_id,
            'tahun_ajaran_id' => $request->tahun_ajaran_id,
        ]);

        return redirect()->back()->with('success', 'Data Kelas berhasil diperbarui.');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Terjadi kesalahan saat menyimpan data.');
    }
}


public function deleteKelas($id)
{
    $kelas = Kelas::findOrFail($id);
    $kelas->delete();

    return redirect()->back()->with('success', 'Data Kelas berhasil dihapus.');
}

public function detailKelas(Request $request)
{
    $query = Kelas::query();

    if ($request->filled('kelas') && $request->filled('tahun_awal') && $request->filled('tahun_akhir')) {
        $kelasInput = $request->kelas;

        $query->where('nama', 'like', $kelasInput . '%')
              ->whereHas('semester.tahunAjaran', function ($q) use ($request) {
                  $q->where('tahun_awal', $request->tahun_awal)
                    ->where('tahun_akhir', $request->tahun_akhir);
              });
    }

    // Ambil data kelas beserta semua relasi
    $kelas = $query->with([
        'waliKelas',
        'semester.tahunAjaran',
        'kelasMapel.mapel',
        'kelasMapel.guru.user',
        'siswaKelas.siswa.user'
    ])->orderBy('created_at', 'desc')->get();

    // Ambil data tambahan untuk dropdown
    $guru = User::where('role', 'guru')->withTrashed()->get();
    $semester = Semester::with('tahunAjaran')->get();
    $tahun = TahunAjaran::all();
    $allKelas = Kelas::with(['waliKelas', 'semester.tahunAjaran'])->get();
    $mapels = Mapel::whereNull('deleted_at')
        ->orderBy('created_at', 'desc')->get();

    $gurus = Guru::whereNull('deleted_at')
        ->whereHas('user', fn ($q) => $q->whereNull('deleted_at'))
        ->with('user')
        ->orderBy('created_at', 'desc')
        ->get();

    $siswas = Siswa::whereNull('deleted_at')
        ->whereHas('user', fn ($q) => $q->whereNull('deleted_at'))
        ->with('user')
        ->orderBy('created_at', 'desc')
        ->get();


    // filter dropdown kelas
$filterOptions = Kelas::with(['semester.tahunAjaran' => function ($query) {
        $query->withTrashed();
    }])
    ->get()
    ->filter(function ($item) {
        return $item->semester && $item->semester->tahunAjaran && !$item->semester->tahunAjaran->trashed();
    })
    ->map(function ($item) {
        preg_match('/\d+/', $item->nama, $matches);
        $kelasNumber = $matches[0] ?? $item->nama;

        return [
            'nama' => $kelasNumber,
            'tahun_awal' => $item->semester->tahunAjaran->tahun_awal,
            'tahun_akhir' => $item->semester->tahunAjaran->tahun_akhir,
        ];
    })
    ->unique(function ($item) {
        return $item['nama'] . '-' . $item['tahun_awal'] . '-' . $item['tahun_akhir'];
    })
    ->values();

    return view('backend.admin.detail_kelas', compact(
        'kelas', 'guru', 'semester', 'tahun',
        'mapels', 'gurus', 'siswas', 'allKelas', 'filterOptions'
    ));
}

public function importDetailKelas(Request $request)
{
    $request->validate([
        'kelas_id' => 'required|exists:kelas,id',
        'file_guru_mapel' => 'nullable|file|mimes:xlsx,xls,csv',
        'file_siswa' => 'nullable|file|mimes:xlsx,xls,csv',
    ]);

    $kelas = Kelas::findOrFail($request->kelas_id);

    // Validasi jika sudah ada guru mapel di kelas
    if ($request->hasFile('file_guru_mapel')) {
        $sudahAdaGuruMapel = KelasMapel::where('kelas_id', $kelas->id)->exists();
        if ($sudahAdaGuruMapel) {
            return redirect()->back()->with('error', "Data guru mapel untuk kelas {$kelas->nama} sudah ada");
        }
    }

    // Validasi jika sudah ada siswa di kelas
    if ($request->hasFile('file_siswa')) {
        $sudahAdaSiswa = SiswaKelas::where('kelas_id', $kelas->id)->exists();
        if ($sudahAdaSiswa) {
            return redirect()->back()->with('error', "Data siswa untuk kelas {$kelas->nama} sudah ada");
        }
    }

    $statusMessage = '';

    DB::transaction(function () use ($request, $kelas, &$statusMessage) {
        $mapelAdded = false;
        $siswaAdded = false;

        // Import Guru Mapel
        if ($request->hasFile('file_guru_mapel')) {
            Excel::import(new GuruMapelImport($kelas->id), $request->file('file_guru_mapel'));
            $mapelAdded = true;

            // Ambil hanya kelas_mapel dengan mapel tipe = 1 (penilaian)
            $kelasMapelIds = KelasMapel::where('kelas_id', $kelas->id)
                ->whereHas('mapel', fn($q) => $q->where('tipe', 1))
                ->pluck('id');

            $siswaLama = SiswaKelas::where('kelas_id', $kelas->id)->pluck('siswa_id');

            foreach ($siswaLama as $siswaId) {
                foreach ($kelasMapelIds as $kelasMapelId) {
                    Nilai::firstOrCreate([
                        'siswa_id' => $siswaId,
                        'kelas_mapel_id' => $kelasMapelId,
                    ], [
                        's_nilai' => json_encode(array_fill_keys([
                            'sumatif_1','sumatif_2','sumatif_3','sumatif_4','sumatif_5',
                            'sumatif_6','sumatif_7','sumatif_8','sumatif_9','sumatif_10'
                        ], null)),
                    ]);
                }
            }
        }

        // Import Siswa
        if ($request->hasFile('file_siswa')) {
            Excel::import(new SiswaKelasImport($kelas->id), $request->file('file_siswa'));
            $siswaAdded = true;

            $siswaDiKelas = SiswaKelas::where('kelas_id', $kelas->id)->pluck('siswa_id');

            $kelasMapelIds = KelasMapel::where('kelas_id', $kelas->id)
                ->whereHas('mapel', fn($q) => $q->where('tipe', 1))
                ->pluck('id');

            foreach ($siswaDiKelas as $siswaId) {
                foreach ($kelasMapelIds as $kelasMapelId) {
                    Nilai::firstOrCreate([
                        'siswa_id' => $siswaId,
                        'kelas_mapel_id' => $kelasMapelId,
                    ], [
                        's_nilai' => json_encode(array_fill_keys([
                            'sumatif_1','sumatif_2','sumatif_3','sumatif_4','sumatif_5',
                            'sumatif_6','sumatif_7','sumatif_8','sumatif_9','sumatif_10'
                        ], null)),
                    ]);
                }
            }
        }

        if ($mapelAdded && $siswaAdded) {
            $statusMessage = "Import data guru mapel dan siswa berhasil ditambahkan ke kelas {$kelas->nama}.";
        } elseif ($mapelAdded) {
            $statusMessage = "Import data guru mapel berhasil ditambahkan ke kelas {$kelas->nama}.";
        } elseif ($siswaAdded) {
            $statusMessage = "Import data siswa berhasil ditambahkan ke kelas {$kelas->nama}.";
        } else {
            $statusMessage = "Tidak ada data yang diimport ke kelas {$kelas->nama}.";
        }
    });

    return redirect()->back()->with('success', $statusMessage);
}


public function storeDetailKelas(Request $request)
{
    $request->validate([
        'kelas_id' => 'required|exists:kelas,id',
    ]);

    $kelas = Kelas::findOrFail($request->kelas_id);

    // Cek apakah mapel sudah pernah ditambahkan ke kelas
    if (is_array($request->mapel_id)) {
        foreach ($request->mapel_id as $mapel_id) {
            $sudahAda = KelasMapel::where('kelas_id', $kelas->id)
                ->where('mapel_id', $mapel_id)
                ->exists();

            if ($sudahAda) {
                $namaMapel = optional(\App\Models\Mapel::find($mapel_id))->nama ?? 'Mapel';
                return redirect()->back()->withErrors([
                    'duplicate_mapel' => "Mapel $namaMapel sudah pernah ditambahkan ke kelas {$kelas->nama}"
                ]);
            }
        }
    }

    // Cek apakah siswa sudah ada di kelas
    if (is_array($request->siswa_id)) {
        foreach ($request->siswa_id as $siswa_id) {
            $sudahAda = \App\Models\SiswaKelas::where('siswa_id', $siswa_id)
                ->where('kelas_id', $kelas->id)
                ->exists();

            if ($sudahAda) {
                $nama = optional(\App\Models\User::find($siswa_id))->name ?? 'Siswa';
                return redirect()->back()->withErrors([
                    'duplicate_siswa' => "Siswa $nama sudah terdaftar di kelas {$kelas->nama}."
                ]);
            }
        }
    }

    DB::transaction(function () use ($request, $kelas, &$statusMessage) {
        $mapelAdded = false;
        $siswaAdded = false;

        // Simpan ke kelas_mapel
        if (is_array($request->mapel_id) && is_array($request->guru_id)) {
            foreach ($request->mapel_id as $i => $mapel_id) {
                $guru_id = $request->guru_id[$i] ?? null;

                \App\Models\KelasMapel::create([
                    'kelas_id' => $kelas->id,
                    'mapel_id' => $mapel_id,
                    'guru_id' => $guru_id,
                ]);

                $mapelAdded = true;
            }

            // Nilai default untuk siswa lama
            $siswaLama = \App\Models\SiswaKelas::where('kelas_id', $kelas->id)->pluck('siswa_id');
            $penilaianMapelIds = \App\Models\Mapel::whereIn('id', $request->mapel_id)
                ->where('tipe', 1)
                ->pluck('id');

            $kelasMapelPenilaian = \App\Models\KelasMapel::where('kelas_id', $kelas->id)
                ->whereIn('mapel_id', $penilaianMapelIds)
                ->pluck('id');

            foreach ($siswaLama as $siswa_id) {
                foreach ($kelasMapelPenilaian as $kelasMapelId) {
                    \App\Models\Nilai::firstOrCreate([
                        'siswa_id' => $siswa_id,
                        'kelas_mapel_id' => $kelasMapelId,
                    ], [
                        's_nilai' => json_encode(array_fill_keys([
                            'sumatif_1', 'sumatif_2', 'sumatif_3', 'sumatif_4', 'sumatif_5',
                            'sumatif_6', 'sumatif_7', 'sumatif_8', 'sumatif_9', 'sumatif_10'
                        ], null)),
                    ]);
                }
            }
        }

        // Simpan ke siswa_kelas
        if (is_array($request->siswa_id)) {
            foreach ($request->siswa_id as $siswa_id) {
                \App\Models\SiswaKelas::firstOrCreate([
                    'siswa_id' => $siswa_id,
                    'kelas_id' => $kelas->id,
                ]);
                $siswaAdded = true;
            }

            $kelasMapelPenilaian = \App\Models\KelasMapel::where('kelas_id', $kelas->id)
                ->whereHas('mapel', fn ($q) => $q->where('tipe', 1))
                ->get();

            foreach ($request->siswa_id as $siswa_id) {
                foreach ($kelasMapelPenilaian as $kelasMapel) {
                    \App\Models\Nilai::firstOrCreate([
                        'siswa_id' => $siswa_id,
                        'kelas_mapel_id' => $kelasMapel->id,
                    ], [
                        's_nilai' => json_encode(array_fill_keys([
                            'sumatif_1', 'sumatif_2', 'sumatif_3', 'sumatif_4', 'sumatif_5',
                            'sumatif_6', 'sumatif_7', 'sumatif_8', 'sumatif_9', 'sumatif_10'
                        ], null)),
                    ]);
                }
            }
        }

        if ($mapelAdded && $siswaAdded) {
            $statusMessage = "Data guru mapel dan siswa berhasil ditambahkan ke kelas {$kelas->nama}.";
        } elseif ($mapelAdded) {
            $statusMessage = "Data guru mapel berhasil ditambahkan ke kelas {$kelas->nama}.";
        } elseif ($siswaAdded) {
            $statusMessage = "Data siswa berhasil ditambahkan ke kelas {$kelas->nama}.";
        } else {
            $statusMessage = "Tidak ada data yang ditambahkan ke kelas {$kelas->nama}.";
        }
    });

    return redirect()->back()->with('success', $statusMessage);
}

public function duplikatDetailKelas()
{
    try {
        $kelasSemester1 = Kelas::with('semester')
            ->whereHas('semester', function ($query) {
                $query->where('semester', 1);
            })->get();

        if ($kelasSemester1->isEmpty()) {
            throw new \Exception("Semua kelas sudah berada di semester 2. Tidak ada data untuk diduplikat.");
        }

        $jumlahDuplikat = 0;

        DB::transaction(function () use ($kelasSemester1, &$jumlahDuplikat) {
            foreach ($kelasSemester1 as $kelas) {
                // Cek atau buat semester 2
                $semester2 = Semester::firstOrCreate([
                    'semester' => 2,
                    'tahun_ajaran_id' => $kelas->tahun_ajaran_id,
                ]);

                // Duplikat kelas
                $kelasBaru = Kelas::firstOrCreate([
                    'nama' => $kelas->nama,
                    'wali_kelas_id' => $kelas->wali_kelas_id,
                    'semester_id' => $semester2->id,
                    'tahun_ajaran_id' => $kelas->tahun_ajaran_id,
                ]);

                if ($kelasBaru->wasRecentlyCreated) {
                    $jumlahDuplikat++;
                }

                // Duplikat KelasMapel
                $kelasMapelsLama = KelasMapel::with('mapel')
                    ->where('kelas_id', $kelas->id)
                    ->get();

                $kelasMapelMapping = [];

                foreach ($kelasMapelsLama as $mapelLama) {
                    $mapelBaru = KelasMapel::firstOrCreate([
                        'kelas_id' => $kelasBaru->id,
                        'mapel_id' => $mapelLama->mapel_id,
                        'guru_id' => $mapelLama->guru_id,
                    ]);

                    $kelasMapelMapping[$mapelLama->id] = [
                        'id' => $mapelBaru->id,
                        'tipe' => $mapelLama->mapel->tipe ?? 1,
                    ];
                }

                // Duplikat Siswa dan Nilai
                $siswaKelas = SiswaKelas::where('kelas_id', $kelas->id)->get();

                foreach ($siswaKelas as $siswa) {
                    SiswaKelas::firstOrCreate([
                        'kelas_id' => $kelasBaru->id,
                        'siswa_id' => $siswa->siswa_id,
                    ]);

                    foreach ($kelasMapelMapping as $info) {
                        if ($info['tipe'] == 1) {
                            Nilai::firstOrCreate([
                                'siswa_id' => $siswa->siswa_id,
                                'kelas_mapel_id' => $info['id'],
                            ], [
                                's_nilai' => json_encode(array_fill_keys([
                                    'sumatif_1', 'sumatif_2', 'sumatif_3', 'sumatif_4', 'sumatif_5',
                                    'sumatif_6', 'sumatif_7', 'sumatif_8', 'sumatif_9', 'sumatif_10'
                                ], null)),
                            ]);
                        }
                    }
                }
            }
        });

        return redirect()->back()->with('success', "Berhasil menduplikat kelas ke semester 2.");
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Gagal duplikat kelas: ' . $e->getMessage());
    }
}


public function exportGuruMapel($kelas_id)
{
    $kelas = Kelas::with('semester.tahunAjaran')->findOrFail($kelas_id);

    $namaKelas   = $kelas->nama;
    $semester    = $kelas->semester->semester ?? 'x';
    $tahunAwal   = $kelas->semester->tahunAjaran->tahun_awal ?? 'xxxx';
    $tahunAkhir  = $kelas->semester->tahunAjaran->tahun_akhir ?? 'yyyy';

    $fileName = "data_guru-mapel_kelas-{$namaKelas}_semester-{$semester}_{$tahunAwal}-{$tahunAkhir}.xlsx";

    return Excel::download(new GuruMapelExport($kelas_id), $fileName);
}

public function deleteKelasMapel($id)
{
    $kelasMapel = KelasMapel::findOrFail($id);

    Nilai::where('kelas_mapel_id', $id)->delete();

    $kelasMapel->delete();

    return redirect()->back()->with('success', 'Data Guru Mapel berhasil dihapus.');
}
public function deleteSiswaKelas($id)
{
    $siswaKelas = SiswaKelas::findOrFail($id);

    $siswaId = $siswaKelas->siswa_id;

    Nilai::where('siswa_id', $siswaId)->delete();

    $siswaKelas->delete();

    return redirect()->back()->with('success', 'Data Siswa Kelas berhasil dihapus.');
}


public function deleteDetailKelas($id)
{
    $kelas = Kelas::findOrFail($id);

    $siswaKelasList = $kelas->siswaKelas;

    foreach ($siswaKelasList as $siswaKelas) {
        $siswa = Siswa::where('siswa_id', $siswaKelas->siswa_id)->first();
        $user  = User::where('id', $siswaKelas->siswa_id)->first();

        if ($siswa) {
            $siswa->delete();
        }

        if ($user) {
            $user->delete();
        }
    }

    $kelas->delete();

    return redirect()->back()->with('success', 'Data Kelas dan siswa di dalamnya berhasil dinonaktifkan.');
}

public function deleteFiltered(Request $request)
{
    $ids = $request->input('kelas_ids', []);

    if (empty($ids)) {
        return redirect()->back()->with('error', 'Tidak ada data kelas yang dipilih untuk dinonaktifkan');
    }

    foreach ($ids as $id) {
        $kelas = Kelas::find($id);
        if (!$kelas) continue;

        $siswaKelasList = $kelas->siswaKelas;

        foreach ($siswaKelasList as $siswaKelas) {
            $siswa = Siswa::where('siswa_id', $siswaKelas->siswa_id)->first();
            $user  = User::where('id', $siswaKelas->siswa_id)->first();

            if ($siswa) {
                $siswa->delete();
            }

            if ($user) {
                $user->delete();
            }
        }

        $kelas->delete();
    }

    return redirect()->back()->with('success', 'Semua data detailkelas berhasil dinonaktifkan');
}


public function restoreViewKelas()
{
    $deletedKelas = Kelas::onlyTrashed()->with([
        'waliKelas', 
        'semester.tahunAjaran',
        'kelasMapel.mapel',
        'kelasMapel.guru.user',
        'siswaKelas.siswa.user'
        ])->orderBy('created_at', 'desc')->get();


    $guru = User::where('role', 'guru')->get();
    $semester = Semester::with('tahunAjaran')->get();
    $tahun = TahunAjaran::all();

    $mapels = Mapel::all();
    $gurus = Guru::with('user')->get();
    $siswas = Siswa::with('user')->get();

    $allKelas = Kelas::with(['waliKelas', 'semester.tahunAjaran'])->get();

    return view('backend.admin.kelas_restore', compact(
        'deletedKelas', 'guru', 'semester', 'tahun',
        'mapels', 'gurus', 'siswas', 'allKelas'
    ));
}

public function restoreKelas($id)
{
    $kelas = Kelas::onlyTrashed()->where('id', $id)->firstOrFail();

    $kelas->restore();

    $siswaIds = \App\Models\SiswaKelas::where('kelas_id', $kelas->id)
        ->pluck('siswa_id');

    \App\Models\Siswa::withTrashed()
        ->whereIn('siswa_id', $siswaIds)
        ->restore();

    \App\Models\User::withTrashed()
        ->whereIn('id', $siswaIds)
        ->restore();

    return redirect()->route('admin.kelas.detailKelas')
        ->with('success', 'Kelas dan siswa berhasil dipulihkan.');
}

public function deleteKelasPermanent($id)
{
    $kelas = Kelas::onlyTrashed()->findOrFail($id);
    $kelas->forceDelete();

    return redirect()->route('admin.kelas.restoreViewKelas')->with('success', 'Kelas berhasil dihapus permanen.');
}

public function toggleDaftarUlang(Request $request)
{
    // Cek apakah siswa ada
    $jumlahSiswa = Siswa::count();
    if ($jumlahSiswa < 1) {
        return redirect()->back()->with('error', 'Tidak ada siswa yang tersedia.');
    }

    if (session('daftar_ulang')) {
        // OFF: hapus session & kosongkan field
        Session::forget('daftar_ulang');

        Siswa::query()->update([
            'data_berubah' => null,
            'deskripsi' => null,
        ]);

        return redirect()->back()->with('success', 'Daftar Ulang dimatikan');
    } else {
        // ON: aktifkan daftar ulang
        Session::put('daftar_ulang', true);

        return redirect()->back()->with('success', 'Daftar Ulang diaktifkan.');
    }
}


}