<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Guru;
use App\Models\KelasMapel;
use App\Models\Kelas;
use App\Models\JadwalLayanan;
use App\Models\LayananKonseling;
use App\Models\Siswa;
use App\Models\LayananParenting;
use App\Models\BulanKehadiran;
use App\Models\KehadiranSiswa;
use App\Models\SiswaKelas;
use App\Imports\KehadiranSiswaImport;
use App\Exports\KehadiranSiswaExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class BkController extends Controller
{
    /**
     * Display a listing of the resource.
     */
public function index() 
{
    $user = Auth::user();

    if (!$user->guru) {
        return view('backend.bk.index', [
            'layanans_parenting' => collect([])
        ]);
    }

    $guru = $user->guru;

    if (!$guru->guru_id) {
        return view('backend.bk.index', [
            'layanans_parenting' => collect([])
        ]);
    }

    $layanans_parenting = LayananParenting::with(['siswa', 'guru'])
        ->where('guru_id', $guru->guru_id)
        ->where('tipe', 1)
        ->latest()
        ->get();

    return view('backend.bk.index', compact('layanans_parenting'));
}


public function nilai(Request $request)
{
    $user = Auth::user();

    // Pastikan role adalah guru_bk
    if ($user->role !== 'guru_bk') {
        abort(403, 'Akses ditolak: Anda bukan guru BK.');
    }

    // Ambil data guru berdasarkan relasi guru_id = user_id
    $guru = Guru::with('user')
        ->where('guru_id', $user->id)
        ->first();

    if (!$guru) {
        abort(404, 'Data guru BK tidak ditemukan.');
    }

    $kelasParam = $request->kelas;
    $kelases = collect();
    $selectedKey = null;

    // Ambil semua kelas_mapel_id yang diampu oleh guru bk
    $kelasMapelIds = KelasMapel::where('guru_id', $guru->guru_id)
        ->pluck('kelas_id')
        ->unique();

    // Ambil semua kelas dari kelasMapel tersebut + tahun ajaran
    $allKelas = Kelas::whereIn('id', $kelasMapelIds)
        ->with('tahunAjaran')
        ->get()
        ->sortByDesc(fn($k) => $k->tahunAjaran->tahun_awal ?? 0);

    $tahunAjaranTerbaru = optional($allKelas->first())->tahun_ajaran_id;

    // Filter dropdownKelas berdasarkan tahun ajaran terbaru
    $dropdownKelas = $allKelas
        ->filter(fn($k) => $k->tahun_ajaran_id == $tahunAjaranTerbaru)
        ->unique(fn($k) => $k->nama . '-' . $k->tahun_ajaran_id);

    // Ambil kelases berdasarkan parameter
    if ($kelasParam) {
        if (is_numeric($kelasParam)) {
            $kelas = Kelas::find($kelasParam);
            $nama = $kelas->nama;
            $tahunAjaranId = $kelas->tahun_ajaran_id;
            $selectedKey = $nama . '-' . $tahunAjaranId;
        } else {
            [$nama, $tahunAjaranId] = explode('-', $kelasParam);
            $selectedKey = $kelasParam;
        }

        $kelases = Kelas::with([
    'semester',
    'tahunAjaran',
    'kelasMapel.mapel',
    'kelasMapel.guru.user',
    'kelasMapel.nilai',
    'siswaKelas.siswa.user',
])
->where('nama', $nama)
->where('tahun_ajaran_id', $tahunAjaranId)
->whereIn('id', $kelasMapelIds) // boleh tetap pakai ini karena dropdown udah dibatasi
->orderByDesc('semester_id')
->get();

    }

    return view('backend.bk.nilai', compact(
        'guru', 'dropdownKelas', 'kelases', 'selectedKey'
    ));
}


public function profil()
{
    $user = auth()->user();

    if (!$user) {
        abort(403, 'Anda belum login.');
    }

    $guru = Guru::with('user')->where('guru_id', $user->id)->first();

    if (!$guru) {
        abort(404, 'Data guru tidak ditemukan.');
    }

    $mapelBK = KelasMapel::with('mapel')
        ->where('guru_id', $guru->guru_id)
        ->first();

    return view('backend.bk.profil', [
        'guru' => $guru,
        'mapelBK' => $mapelBK,
    ]);
}

public function layanan()
{
    $guru = Auth::user()->guru;

    $layanans = LayananKonseling::with([
            'siswa' => function ($q) {
                $q->withTrashed()->with(['user' => function ($u) {
                    $u->withTrashed();
                }]);
            },
            'jadwal'
        ])
        ->whereHas('jadwal', function ($q) use ($guru) {
            $q->where('guru_id', $guru->guru_id);
        })
        ->latest()
        ->get();

    foreach ($layanans as $layanan) {
        $layanan->sudah_ada_parenting = LayananParenting::where('siswa_id', optional($layanan->siswa->user)->id)
            ->where('tanggal', $layanan->tanggal)
            ->exists();
    }

    $layanans_parenting = LayananParenting::with([
        'siswa' => function ($q) {
            $q->withTrashed();
        },
        'guru'
    ])
    ->where('guru_id', $guru->guru_id)
    ->where('tipe', 1)
    ->latest()
    ->get();

    foreach ($layanans_parenting as $layanan) {
        $siswaUser = $layanan->siswa;
        $siswaModel = Siswa::withTrashed()->where('siswa_id', $siswaUser->id)->first();
        $layanan->siswa_model = $siswaModel;
    }

    return view('backend.bk.layanan', compact('layanans', 'layanans_parenting'));
}


public function storeHasilLayanan(Request $request)
{
    $request->validate([
        'siswa_id'     => 'required|exists:users,id',
        'guru_id'      => 'required|exists:users,id',
        'tanggal'      => 'required|date',
        'tipe'         => 'required|in:0,1',
        'permasalahan' => 'required|string',
        'penyelesaian' => 'required|string',
        'hasil'        => 'required|string',
    ]);

    try {
        $data = $request->only([
            'siswa_id',
            'guru_id',
            'tanggal',
            'tipe',
            'permasalahan',
            'penyelesaian',
            'hasil',
        ]);

        $insert = LayananParenting::create($data);

        return back()->with('success', 'Hasil layanan berhasil dikirim.');
    } catch (\Exception $e) {
        return back()->with('error', 'Gagal simpan: ' . $e->getMessage());
    }
}

public function updateHasilLayanan(Request $request, $id)
{
    $request->validate([
        'penyelesaian' => 'required|string',
        'hasil' => 'required|string',
    ]);

    $layanan = LayananParenting::findOrFail($id);

    $layanan->penyelesaian = $request->penyelesaian;
    $layanan->hasil = $request->hasil;
    $layanan->save();

    return redirect()->back()->with('sukses', 'Data layanan parenting berhasil diperbarui.');
}

public function updateLayanan(Request $request, $id)
{
    $layanan = LayananKonseling::findOrFail($id);
    $layanan->status = $request->status;
    $layanan->save();

    return redirect()->back()->with('success', 'Status layanan berhasil diupdate.');
}

public function data_layanan_siswa(Request $request)
    {
        // Ambil semua Siswa
        $siswas = Siswa::with('user')->get();

        $query = LayananParenting::with([
            'siswa' => function ($q) {
                $q->withTrashed()->with(['siswa' => function ($qq) {
                    $qq->withTrashed();
                }]);
            },
            'guru'
        ]);

        // Filter berdasarkan tipe
        if ($request->has('tipe') && in_array($request->tipe, ['0', '1'])) {
            $query->where('tipe', $request->tipe);
        }

        // Sembunyikan data apabila salah satu field permasalahan/penyelesaian/hasil = null
        $query->whereNotNull('permasalahan')
              ->whereNotNull('penyelesaian')
              ->whereNotNull('hasil');

        $layanans = $query->latest()->get();

        return view('backend.bk.data_layanan_siswa', compact('layanans', 'siswas'));
    }

    public function jadwal()
{
    $guru = Auth::user()->guru;
    $jadwals = JadwalLayanan::where('guru_id', $guru->guru_id)
                ->orderBy('created_at', 'desc')
                ->get();

    return view('backend.bk.jadwal', compact('jadwals'));
}


     public function storeJadwal(Request $request)
    {
        $request->validate([
            'hari' => 'required|string|max:20',
            'jam'  => 'required|string|max:50',
        ]);

        JadwalLayanan::create([
            'guru_id' => Auth::user()->guru->guru_id,
            'hari'    => $request->hari,
            'jam'     => $request->jam,
        ]);

        return redirect()->route('backend.bk.jadwalLayanan')->with('success', 'Jadwal Layanan berhasil ditambahkan.');
    }

    public function updateJadwal(Request $request, $id)
    {
        $request->validate([
            'hari' => 'required|string|max:20',
            'jam'  => 'required|string|max:50',
        ]);

        $jadwal = JadwalLayanan::findOrFail($id);

        if ($jadwal->guru_id !== Auth::user()->guru->guru_id) {
            abort(403);
        }

         // Soft delete data lama
        $jadwal->delete();

        // Buat data baru
        JadwalLayanan::create([
            'guru_id'  => $jadwal->guru_id,
            'siswa_id' => $jadwal->siswa_id,
            'hari'     => $request->hari,
            'jam'      => $request->jam,
        ]);

        return redirect()->route('backend.bk.jadwalLayanan')->with('success', 'Jadwal Layanan berhasil diperbarui.');
    }

    public function deleteJadwal($id)
    {
        $jadwal = JadwalLayanan::findOrFail($id);

        if ($jadwal->guru_id !== Auth::user()->guru->guru_id) {
            abort(403);
        }

        $jadwal->delete();

        return redirect()->back()->with('success', 'Jadwal Layanan berhasil dihapus.');
    }

public function bulanKehadiran()
    {
        $user = Auth::user();

        if ($user->role !== 'guru_bk') {
            abort(403, 'Akses ditolak: Anda bukan guru BK.');
        }

        // Cari record Guru berdasarkan kolom guru_id = user.id
        $guru = Guru::where('guru_id', $user->id)->first();
        if (!$guru) {
            abort(404, 'Data guru tidak ditemukan.');
        }

        // Ambil semua kelas_id dari kelas_mapel yang diajar guru
        $kelasMapelIds = KelasMapel::where('guru_id', $guru->guru_id)
            ->pluck('kelas_id')
            ->unique();

        // Ambil semua kelas yang sesuai
        $allKelas = Kelas::whereIn('id', $kelasMapelIds)
            ->with('tahunAjaran')
            ->get()
            ->sortByDesc(fn($k) => $k->tahunAjaran->tahun_awal ?? 0);

        // Tahun ajaran terbaru
        $tahunAjaranTerbaru = optional($allKelas->first())->tahun_ajaran_id;

        // Filter hanya kelas dari tahun ajaran terbaru
        $kelasList = $allKelas
            ->filter(fn($k) => $k->tahun_ajaran_id == $tahunAjaranTerbaru)
            ->unique('id');

        $bulanList = BulanKehadiran::with('kelas')
            ->where('guru_id', $user->id)   // pakai user.id
            ->orderByDesc('created_at')
            ->get();

        return view('backend.bk.bulan_kehadiran', compact('kelasList', 'bulanList'));
    }

    public function storeBulanKehadiran(Request $request)
{
    $request->validate([
        'kelas_id' => 'required|array',
        'kelas_id.*' => 'exists:kelas,id',
        'bulan' => 'required|string|max:50',
    ]);

    $guruId = Auth::id();
    $bulanInput = $request->bulan;
    $duplikat = [];

    foreach ($request->kelas_id as $kelasId) {
        $cek = BulanKehadiran::where('kelas_id', $kelasId)
            ->where('bulan', $bulanInput)
            ->first();

        if ($cek) {
            $namaKelas = \App\Models\Kelas::find($kelasId)->nama ?? 'Kelas ID ' . $kelasId;
            $duplikat[] = $namaKelas;
        }
    }

    if (!empty($duplikat)) {
        return redirect()->back()
            ->withErrors(['duplicate' => 'Bulan ' . $bulanInput . ' untuk kelas ' . implode(', ', $duplikat) . ' sudah ada.'])
            ->withInput();
    }

    foreach ($request->kelas_id as $kelasId) {
        $bulan = BulanKehadiran::create([
            'guru_id' => $guruId,
            'kelas_id' => $kelasId,
            'bulan' => $bulanInput,
        ]);

        $siswaList = SiswaKelas::where('kelas_id', $kelasId)
            ->with('siswa')
            ->get();

        foreach ($siswaList as $siswa) {
            KehadiranSiswa::create([
                'bulan_kehadiran_id' => $bulan->id,
                'siswa_id' => $siswa->siswa_id,
                'sakit' => null,
                'izin' => null,
                'alpha' => null,
            ]);
        }
    }

    return redirect()->route('backend.bk.bulan-kehadiran')
        ->with('success', 'Bulan kehadiran siswa berhasil ditambahkan.');
}


public function deleteBulanKehadiran($id)
    {
        $user    = Auth::user();
        $guruId  = $user->id;

        // Cari record BulanKehadiran yang sesuai dan milik user yg login
        $bulan = BulanKehadiran::where('id', $id)
            ->where('guru_id', $guruId)
            ->first();

        if (! $bulan) {
            return redirect()
                ->back()
                ->with('error', 'Bulan kehadiran tidak ditemukan atau anda tidak memiliki izin.');
        }

        // Hapus semua entri KehadiranSiswa yang terkait dengan bulan ini
        KehadiranSiswa::where('bulan_kehadiran_id', $bulan->id)->delete();

        $bulan->delete();

        return redirect()
            ->route('backend.bk.bulan-kehadiran')
            ->with('success', 'Bulan kehadiran dan data kehadiran siswa berhasil dihapus.');
    }

    public function kehadiranSiswa(Request $request)
    {
        $user = Auth::user();

        if ($user->role !== 'guru_bk') {
            abort(403, 'Akses ditolak: Anda bukan guru BK.');
        }

        // Ambil data Guru (tabel guru) berdasarkan user.id
        $guru = Guru::where('guru_id', $user->id)->first();
        if (!$guru) {
            abort(404, 'Data guru tidak ditemukan.');
        }

        // Dapatkan semua kelas_id yang diampu oleh guru
        $kelasMapelIds = KelasMapel::where('guru_id', $guru->guru_id)
            ->pluck('kelas_id')
            ->unique();

        // Ambil data Kelas beserta relasi TahunAjaran, Semester
        $allKelas = Kelas::whereIn('id', $kelasMapelIds)
            ->with(['tahunAjaran', 'semester'])
            ->get()
            ->sortByDesc(fn($k) => $k->tahunAjaran->tahun_awal ?? 0);

        // Ambil tahun ajaran terbaru
        $tahunAjaranTerbaru = optional($allKelas->first())->tahun_ajaran_id;

        // Buat dropdownKelas
        $dropKeys = [];
        foreach ($allKelas as $kelas) {
            if ($kelas->tahun_ajaran_id === $tahunAjaranTerbaru) {
                $key = $kelas->nama . '-' . $kelas->tahun_ajaran_id;
                if (! in_array($key, $dropKeys, true)) {
                    $dropKeys[] = $key;
                }
            }
        }

        $dropdownKelas = collect($dropKeys)->map(function($item) {
            [$nama, $tahunId] = explode('-', $item);
            return [
                'nama'            => $nama,
                'tahun_ajaran_id' => (int) $tahunId,
            ];
        });

        // Jika parameter kelas ada, proses
        $kelasParam  = $request->kelas;   // misal: "7A-1"
        $selectedKey = null;
        $kelases     = collect();
        $kehadiranAll= collect();

        if ($kelasParam) {
            [$nama, $tahunId] = explode('-', $kelasParam);
            $selectedKey = $kelasParam;

            $kelases = Kelas::with(['tahunAjaran', 'semester'])
                ->where('nama', $nama)
                ->where('tahun_ajaran_id', $tahunId)
                ->whereIn('id', $kelasMapelIds)
                ->orderByDesc('semester_id')
                ->get();

            // Untuk setiap $kelasObj, ambil semua BulanKehadiran
            foreach ($kelases as $kelasObj) {

                // Cari semua entri BulanKehadiran untuk kelas ini, untuk guru yang login
                $bulanEntries = BulanKehadiran::where('kelas_id', $kelasObj->id)
                    ->where('guru_id', $user->id)
                    ->orderByDesc('created_at')
                    ->get();

                $bulanList = collect();

                foreach ($bulanEntries as $bulanK) {

                    // Ambil data kehadiran siswa untuk entri bulan ini
                    $kehadiran = KehadiranSiswa::with([
                        'siswa',
                        'siswa.siswa'   // User → Siswa untuk NISN
                    ])
                    ->where('bulan_kehadiran_id', $bulanK->id)
                    ->get();

                    $bulanList[$bulanK->id] = [
                        'model'     => $bulanK,
                        'kehadiran' => $kehadiran,
                    ];
                }

                // Simpan ke koleksi keyed by kelas_id
                $kehadiranAll[$kelasObj->id] = [
                    'kelas'     => $kelasObj,
                    'bulanList' => $bulanList,
                ];
            }
        }

        return view('backend.bk.kehadiran_siswa', compact(
            'dropdownKelas',
            'selectedKey',
            'kelases',
            'kehadiranAll'
        ));
    }

    public function updateKehadiranSiswa(Request $request, $id)
{
    $request->validate([
        'sakit' => 'required|integer|min:0',
        'izin'  => 'required|integer|min:0',
        'alpha' => 'required|integer|min:0',
    ]);

    $kehadiran = KehadiranSiswa::findOrFail($id);

    $kehadiran->update([
        'sakit' => $request->sakit,
        'izin'  => $request->izin,
        'alpha' => $request->alpha,
    ]);

    return back()->with('success', 'Data kehadiran berhasil diperbarui.');
}

public function exportKehadiranSiswa($bulanKehadiranId)
{
    $bulan = BulanKehadiran::with('kelas.tahunAjaran', 'kelas.semester')
        ->findOrFail($bulanKehadiranId);

    $kelasNama     = Str::slug($bulan->kelas->nama);
    $semester      = $bulan->kelas->semester->semester;
    $tahunAwal     = $bulan->kelas->tahunAjaran->tahun_awal;
    $tahunAkhir    = $bulan->kelas->tahunAjaran->tahun_akhir;
    $namaBulan     = \Carbon\Carbon::parse($bulan->bulan)->translatedFormat('F-Y');
    $slugBulan     = Str::slug($namaBulan); 

    $filename = "data-kehadiran_bulan-{$slugBulan}_kelas-{$kelasNama}_semester-{$semester}_{$tahunAwal}-{$tahunAkhir}.xlsx";

    return Excel::download(new KehadiranSiswaExport($bulanKehadiranId), $filename);
}

public function importKehadiranSiswa(Request $request, $bulanKehadiranId)
{
    $request->validate([
        'file' => 'required|mimes:xlsx,xls'
    ]);

    try {
        Excel::import(new KehadiranSiswaImport($bulanKehadiranId), $request->file('file'));

        return back()->with('success', 'Data berhasil diimpor.');
    } catch (ValidationException $e) {
        return back()->withErrors(['import_error' => $e->getMessage()]);
    } catch (\Throwable $e) {
        // Fallback error umum selain validasi
        return back()->withErrors(['import_error' => 'Terjadi kesalahan saat impor. Pastikan format file sesuai.']);
    }
}

}


