<?php

namespace App\Http\Controllers;

use App\Models\Guru;
use Illuminate\Http\Request;
use App\Models\TahunAjaran;
use App\Models\Kelas;
use App\Models\Mapel;
use App\Models\KehadiranSiswa;
use App\Models\BulanKehadiran;

class KepsekController extends Controller
{

public function index() 
{
    // Ambil Tahun Ajaran Terbaru
    $ta = TahunAjaran::orderByDesc('tahun_awal')->first();
    
    // Jika tidak ada tahun ajaran, return dengan data kosong
    if (!$ta) {
        return view('backend.kepsek.index', [
            'top' => null,
            'latestBulan' => null,
        ]);
    }

    // Ambil bulan terbaru dari seluruh entri BulanKehadiran
    $latestBulanValue = BulanKehadiran::orderByDesc('bulan')->value('bulan');
    
    // Jika tidak ada data bulan kehadiran, return dengan data kosong
    if (!$latestBulanValue) {
        return view('backend.kepsek.index', [
            'top' => null,
            'latestBulan' => null,
        ]);
    }

    // Ambil semua BulanKehadiran dengan bulan terbaru
    $latestBulanList = BulanKehadiran::with('kehadiranSiswa')
        ->where('bulan', $latestBulanValue)
        ->get();
    
    // Jika tidak ada data bulan kehadiran untuk bulan terbaru, return dengan data kosong
    if ($latestBulanList->isEmpty()) {
        return view('backend.kepsek.index', [
            'top' => null,
            'latestBulan' => $latestBulanValue,
        ]);
    }

    // Ambil semua kelas di tahun ajaran terbaru
    $kelasList = Kelas::with(['semester', 'tahunAjaran', 'waliKelas', 'siswaKelas'])
        ->where('tahun_ajaran_id', $ta->id)
        ->get();
    
    // Jika tidak ada kelas di tahun ajaran terbaru, return dengan data kosong
    if ($kelasList->isEmpty()) {
        return view('backend.kepsek.index', [
            'top' => null,
            'latestBulan' => $latestBulanValue,
        ]);
    }

    // Hitung total alpha per kelas dari semua entri bulan terbaru
    $kelasWithAlpha = $kelasList->map(function($kelas) use ($latestBulanList) {
        $siswaIds = $kelas->siswaKelas->pluck('siswa_id');

        // Ambil total alpha dari seluruh entri BulanKehadiran di bulan tersebut
        $totalAlpha = $latestBulanList->sum(function ($bulan) use ($siswaIds) {
            return KehadiranSiswa::where('bulan_kehadiran_id', $bulan->id)
                ->whereIn('siswa_id', $siswaIds)
                ->sum('alpha');
        });

        return (object) [
            'kelas'       => $kelas,
            'total_alpha' => $totalAlpha,
        ];
    });

    // Jika tidak ada data kelas dengan alpha, return dengan data kosong
    if ($kelasWithAlpha->isEmpty()) {
        return view('backend.kepsek.index', [
            'top' => null,
            'latestBulan' => $latestBulanValue,
        ]);
    }

    // Pilih kelas dengan total alpha terbanyak
    $top = $kelasWithAlpha->sortByDesc('total_alpha')->first();

    return view('backend.kepsek.index', [
        'top' => $top,
        'latestBulan' => $latestBulanValue,
    ]);
}

    public function profil()
{
    $user = auth()->user();

    if ($user->role !== 'kepala_sekolah') {
        abort(403, 'Akses ditolak');
    }

    $guru = Guru::with('user')
        ->where('guru_id', $user->id)
        ->get();

    return view('backend.kepsek.profil', compact('guru'));
}


public function nilaiKelas()
    {
        // Ambil tahun ajaran terbaru
        $tahunAjaranTerbaru = TahunAjaran::orderBy('tahun_awal', 'desc')->first();
        
        if (!$tahunAjaranTerbaru) {
            return view('backend.kepala_sekolah.nilai_kelas', [
                'kelasOptions' => collect(),
                'mapelOptions' => collect(),
                'selectedData' => collect()
            ]);
        }

        // Ambil kelas unik berdasarkan nama
        $kelasOptions = Kelas::where('tahun_ajaran_id', $tahunAjaranTerbaru->id)
            ->with(['semester', 'tahunAjaran'])
            ->get()
            ->groupBy('nama')
            ->map(function ($kelasGroup) {
                return [
                    'nama' => $kelasGroup->first()->nama,
                    'semesters' => $kelasGroup->map(function ($kelas) {
                        return [
                            'id' => $kelas->id,
                            'semester' => $kelas->semester->semester,
                            'label' => $kelas->nama . ' - Semester ' . $kelas->semester->semester
                        ];
                    })
                ];
            });

        // Ambil semua mapel
        $mapelOptions = Mapel::where('tipe', 1) // hanya mapel yang ada penilaian
            ->orderBy('nama')
            ->get(['id', 'nama']);

        return view('backend.kepsek.nilai', compact('kelasOptions', 'mapelOptions'));
    }

    public function cariNilaiKelas(Request $request)
    {
        $kelasIds = $request->input('kelas_ids', []);
        $mapelIds = $request->input('mapel_ids', []);

        if (empty($kelasIds)) {
            return redirect()->back()->with('error', 'Pilih minimal satu kelas!');
        }

        // Ambil data kelas yang dipilih
        $kelases = Kelas::whereIn('id', $kelasIds)
            ->with([
                'semester',
                'tahunAjaran',
                'kelasMapel' => function ($query) use ($mapelIds) {
                    if (!empty($mapelIds)) {
                        $query->whereIn('mapel_id', $mapelIds);
                    }
                    $query->with([
                        'mapel',
                        'guru.user',
                        'nilai'
                    ]);
                },
                'siswaKelas.siswa'
            ])
            ->get();

        // Proses data untuk setiap kelas
        $selectedData = $kelases->map(function ($kelas) {
            $mapelData = $kelas->kelasMapel
    ->filter(fn($km) => $km->mapel->tipe == 1)
    ->map(function ($kelasMapel) use ($kelas) {

    $allNilai = $kelasMapel->nilai;
    $totalSiswa = $kelas->siswaKelas->count();
    
    $nilaiTerlaksana = 0;
    $totalNilaiSemua = [];

    foreach ($allNilai as $nilai) {
        $sNilai = is_string($nilai->s_nilai) 
            ? json_decode($nilai->s_nilai, true) 
            : ($nilai->s_nilai ?? []);

        if (!empty($sNilai)) {
            $nilaiTerisi = array_filter($sNilai, fn($v) => $v !== null);
            $jumlahNilai = count($sNilai); // <- Jumlah rencana = semua array, isi/kosong
            $nilaiTerlaksana = max($nilaiTerlaksana, count($nilaiTerisi));

            foreach ($nilaiTerisi as $n) {
                $totalNilaiSemua[] = $n;
            }
        }
    }

    // Hitung rencana dari salah satu nilai pertama
    $rencanaSumatif = 0;
$firstSNilai = $allNilai->first()->s_nilai ?? null;

if (is_string($firstSNilai)) {
    $decoded = json_decode($firstSNilai, true);
    $rencanaSumatif = is_array($decoded) ? count($decoded) : 0;
} elseif (is_array($firstSNilai)) {
    $rencanaSumatif = count($firstSNilai);
}


    $rataRata = !empty($totalNilaiSemua) 
    ? round(array_sum($totalNilaiSemua) / count($totalNilaiSemua))
    : '-';


    return [
    'mapel_nama' => $kelasMapel->mapel->nama,
    'pengajar_nama' => $kelasMapel->guru->user->name ?? '-',
    'pengajar_nip' => $kelasMapel->guru->nip ?? '-',
    'rencana_sumatif' => $rencanaSumatif,
    'sumatif_terlaksana' => $nilaiTerlaksana,
    'rata_rata' => $rataRata
];

}); 

            return [
                'kelas' => $kelas,
                'mapel_data' => $mapelData
            ];
        });

        // Untuk dropdown
        $tahunAjaranTerbaru = TahunAjaran::orderBy('tahun_awal', 'desc')->first();
        $kelasOptions = Kelas::where('tahun_ajaran_id', $tahunAjaranTerbaru->id)
            ->with(['semester', 'tahunAjaran'])
            ->get()
            ->groupBy('nama')
            ->map(function ($kelasGroup) {
                return [
                    'nama' => $kelasGroup->first()->nama,
                    'semesters' => $kelasGroup->map(function ($kelas) {
                        return [
                            'id' => $kelas->id,
                            'semester' => $kelas->semester->semester,
                            'label' => $kelas->nama . ' - Semester ' . $kelas->semester->semester
                        ];
                    })
                ];
            });

        $mapelOptions = Mapel::where('tipe', 1)->orderBy('nama')->get(['id', 'nama']);

        return view('backend.kepsek.nilai', compact('kelasOptions', 'mapelOptions', 'selectedData'))
            ->with('selectedKelasIds', $kelasIds)
            ->with('selectedMapelIds', $mapelIds);
    }

    public function kehadiran(Request $request)
{
    // Ambil tahun ajaran terbaru
    $ta = TahunAjaran::orderByDesc('tahun_awal')->first();

    // Opsi kelas (group by nama, semesters)
    $kelasOptions = Kelas::where('tahun_ajaran_id', $ta->id)
        ->with('semester', 'tahunAjaran')
        ->get()
        ->groupBy('nama')
        ->map(function($grp) {
            return [
                'nama'      => $grp->first()->nama,
                'semesters' => $grp->map(fn($k) => [
                    'id'       => $k->id,
                    'label'    => "{$k->nama} - Semester {$k->semester->semester}",
                    'semester' => $k->semester->semester,
                ])->sortByDesc('semester')->values(),
            ];
        });

    $selected = $request->input('kelas_ids', []);
    $kelases  = collect();

    if (!empty($selected)) {
        $kelases = Kelas::with([
                'semester',
                'tahunAjaran',
                'waliKelas',
                'bulanKehadiran.kehadiranSiswa.siswa',
                'bulanKehadiran.kehadiranSiswa.siswa.siswa'
            ])
            ->whereIn('id', $selected)
            ->get();
    }

    return view('backend.kepsek.kehadiran', compact('kelasOptions', 'kelases', 'selected'));
}

}