<?php

namespace App\Imports;

use App\Models\Guru;
use App\Models\Semester;
use App\Models\TahunAjaran;
use App\Models\Kelas;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class KelasImport implements ToCollection, WithHeadingRow
{
    public function collection(Collection $rows)
    {
        $this->validateData($rows);

        $successCount = 0;
        $skipCount = 0;
        $skipMessages = [];

        foreach ($rows as $index => $row) {
            $rowNumber = $index + 2;

            try {
                $result = $this->processRow($row, $rowNumber, $skipMessages);
                if ($result) {
                    $successCount++;
                } else {
                    $skipCount++;
                }
            } catch (\Exception $e) {
                Log::error("Error pada baris {$rowNumber}: " . $e->getMessage());
                throw $e;
            }
        }

        if (!empty($skipMessages)) {
            throw new \Exception(implode("; ", $skipMessages));
        }

        Log::info("Import selesai: {$successCount} data berhasil, {$skipCount} data dilewati");
    }

    private function validateData(Collection $rows)
    {
        $errors = [];
        $rowNumber = 2;

        foreach ($rows as $row) {
            $nip = trim($row['nip'] ?? '');
            $tahunAwal = trim($row['tahun_awal'] ?? '');
            $tahunAkhir = trim($row['tahun_akhir'] ?? '');
            $semester = trim($row['semester'] ?? '');
            $nama = trim($row['nama'] ?? '');

            $missingFields = [];
            if (empty($nip)) $missingFields[] = 'nip';
            if (empty($tahunAwal)) $missingFields[] = 'tahun_awal';
            if (empty($tahunAkhir)) $missingFields[] = 'tahun_akhir';
            if (empty($semester)) $missingFields[] = 'semester';
            if (empty($nama)) $missingFields[] = 'nama';

            if (!empty($missingFields)) {
                $errors[] = "Baris {$rowNumber}: kolom " . implode(', ', $missingFields) . " kosong";
                $rowNumber++;
                continue;
            }

            if (!is_numeric($tahunAwal) || !is_numeric($tahunAkhir)) {
                $errors[] = "Baris {$rowNumber}: Tahun harus berupa angka";
                $rowNumber++;
                continue;
            }

            if ((int)$tahunAkhir - (int)$tahunAwal !== 1) {
                $errors[] = "Baris {$rowNumber}: Perbedaan tahun_awal dan tahun_akhir harus tepat 1 tahun";
            }

            if ((int)$tahunAwal >= (int)$tahunAkhir) {
                $errors[] = "Baris {$rowNumber}: tahun_awal tidak boleh lebih besar atau sama dengan tahun_akhir";
            }

            // Cek guru
            $guru = Guru::where('nip', $nip)->first();
            if (!$guru) {
                $errors[] = "Baris {$rowNumber}: NIP guru '{$nip}' tidak ditemukan di sistem";
            }

            // Cek tahun ajaran dan semester
            $tahunAjaran = TahunAjaran::where('tahun_awal', (int)$tahunAwal)
                ->where('tahun_akhir', (int)$tahunAkhir)
                ->first();

            if (!$tahunAjaran) {
                $errors[] = "Baris {$rowNumber}: Tahun ajaran {$tahunAwal}/{$tahunAkhir} tidak ditemukan";
            } else {
                $semesterData = Semester::where('semester', $semester)
                    ->where('tahun_ajaran_id', $tahunAjaran->id)
                    ->first();

                if (!$semesterData) {
                    $errors[] = "Baris {$rowNumber}: Semester {$semester} untuk tahun ajaran {$tahunAwal}/{$tahunAkhir} tidak ditemukan";
                }
            }

            // Validasi apakah NIP sudah jadi wali kelas di tahun ajaran
            $sudahWali = Kelas::where('wali_kelas_id', $guru->guru_id)
                ->where('tahun_ajaran_id', $tahunAjaran->id)
                ->exists();

            if ($sudahWali) {
                $errors[] = "Baris {$rowNumber}: NIP '{$nip}' sudah menjadi wali kelas di tahun ajaran {$tahunAwal}/{$tahunAkhir}";
            }

            $rowNumber++;
        }

        if (!empty($errors)) {
            throw new \Exception(implode("; ", $errors));
        }
    }

    private function processRow($row, $rowNumber, &$skipMessages)
    {
        $guru = Guru::where('nip', $row['nip'])->first();
        if (!$guru) {
            $skipMessages[] = "Baris {$rowNumber}: Guru dengan NIP {$row['nip']} tidak ditemukan.";
            return false;
        }

        $userId = $guru->guru_id;

        $tahunAjaran = TahunAjaran::where('tahun_awal', (int)$row['tahun_awal'])
            ->where('tahun_akhir', (int)$row['tahun_akhir'])
            ->first();

        $semester = Semester::where('semester', $row['semester'])
            ->where('tahun_ajaran_id', $tahunAjaran->id)
            ->first();

        $namaKelas = $this->formatNamaKelas($row['nama']);

        $kelasExist = Kelas::where('nama', $namaKelas)
            ->where('semester_id', $semester->id)
            ->where('tahun_ajaran_id', $tahunAjaran->id)
            ->exists();

        if ($kelasExist) {
            $skipMessages[] = "Baris {$rowNumber}: Kelas '{$namaKelas}' sudah ada";
            return false;
        }

        Kelas::create([
            'nama' => $namaKelas,
            'wali_kelas_id' => $userId,
            'semester_id' => $semester->id,
            'tahun_ajaran_id' => $tahunAjaran->id,
        ]);

        return true;
    }

    private function formatNamaKelas($nama)
    {
        if (preg_match('/^\d+[a-z]$/', $nama)) {
            return strtoupper($nama);
        }
        return $nama;
    }
}
